/* vim: set sw=8 ts=8 si : */
/*
 * Serial line daemon program for shutdown of your server.
 * Circit diagram
 * 5 GND ------- 1K ----- LED ----- 4 DTR
 *                              |
 * 8 CTS --------- pushbutton ---
 *
 * A blinking LED indicates that this program is running.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License.
 * See http://www.gnu.org/copyleft/ for details.
 *
 * Written by Guido Socher guidosocher@fastmail.fm
 *
 */
#define VERINFO "version 0.1"
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>



/* Switch the LEDs on or off */
int setled( int fd, int onoff, int *ledstate){
	if (onoff){
		// on 
		*ledstate |= TIOCM_DTR;
	}else{
		*ledstate &= ~TIOCM_DTR;
	}
	ioctl(fd, TIOCMSET, ledstate);
	return *ledstate;
}


/* get the current state of the push button  and return it. */
int getpushbutton(int fd)
{
	int state;
	ioctl(fd, TIOCMGET, &state);	// read interface 
	if (state & TIOCM_CTS) {
		return (1);
	}
	return (0);
}

void help()
{
	printf("rs232-shutdown-button -- serial line pushbutton and LED control\n\
USAGE: rs232-shutdown-button serial-device\n\
\n\
EXAMPLE:\n\
         rs232-shutdown-button [-h][-n] /dev/ttyUSB0\n\
\n\
Use command setserial -g /dev/tty* to see available tty devices.\n\
You can also run the command dmesg after plugging in the USB to rs232 adapter\n\
to see what the name of the device is.\n\
\n\
OPTIONS: -h this help\n\
         -n do not run \"shutdown -h\". Instead just print a message and exit with code\n\
	    zero. This program blocks until the shutdown button is pressed. It will exit with code\n\
non-zero if it was killed and exit with code 0 if the shutdown button was pressed.\n\
This can then be used by a script to trigger the actual shutdown of the server.\n\
Without the -n option the program still blocks but it will also run shutdown -h.\n\
");
#ifdef VERINFO
	puts(VERINFO);
#endif	 
exit(0); 
}


int main(int argc, char **argv)
{
	int lstate = 0;
	int fd;
	int opt_n=0;
	useconds_t lpause=900000; // 0.9sec
	useconds_t spause=100000; // 0.1sec

	// The following things are used for getopt: 
        extern char *optarg __attribute__ ((unused));
        extern int optind;
        extern int opterr;
	int ch;

	opterr=0; // prevent getopt's own error handling, go to '?'
	if (argc<2) help();
	while ((ch = getopt(argc, argv, "hn")) != -1) {
		switch (ch) {
		case 'h':
			help(); // will exit program
		case 'n':
			opt_n=1;
			break;
		case '?':
			fprintf(stderr, "ERROR: No such option. -h for help.\n");
			exit(1);
		//no default action for case 
		}
	}
	//printf("dbg optind, argc %d, %d\n",optind,argc);
	// no arg given:
	if (optind == argc){
		help();
	}
	// we expect only one arg:
	if (optind != argc -1){
		help();
	}
	// open device 
	fd = open(argv[optind], O_RDWR | O_NDELAY);
	if (fd < 0) {
		fprintf(stderr, "ERROR: can not open \"%s\"\n", argv[optind]);
		exit(2);
	}

	// first set a defined state, all off 
	ioctl(fd, TIOCMSET, &lstate);
	while(1){
		// blinking: switch on the DTR LED
		setled( fd, 1, &lstate);
		usleep(lpause);
		if (getpushbutton(fd)){
			// LED will probably go off when we close the file descriptor or when
			// the kernel cleans it up.
			if (opt_n){
				printf("rs232-shutdown-button: button press, exit 0\n");
			}else{
				if (system("/sbin/shutdown -h now")){
					printf("rs232-shutdown-button: /sbin/shutdown failed\n");
				}
			}
			close(fd);
			return(0);
		}
		// blinking: switch off the DTR LED
		setled( fd,0, &lstate);
		usleep(spause);
	}
	// we never get here 
	return (1);
}
